import sys
import os
import re
import math
import pandas as pd
import duckdb
import xlsxwriter
from PyQt5.QtWidgets import (
    QApplication, QMainWindow, QWidget, QVBoxLayout, QHBoxLayout, QPushButton,
    QFileDialog, QLabel, QLineEdit, QComboBox, QListWidget, QAbstractItemView,
    QProgressBar, QGroupBox, QFormLayout, QMessageBox, QStatusBar, QTabWidget,
    QCheckBox, QRadioButton, QScrollArea, QDialog, QTextEdit, QSpinBox, QColorDialog, QFrame
)
from PyQt5.QtGui import QColor, QPalette
from PyQt5.QtCore import Qt, QThread, pyqtSignal

# --- UTILITY: Custom Color Picker Button ---
class ColorButton(QPushButton):
    def __init__(self, default_color, parent=None):
        super().__init__(parent)
        self.color = default_color
        self.clicked.connect(self.pick_color)
        self.update_style()

    def pick_color(self):
        c = QColorDialog.getColor(self.color, self, "Select Color")
        if c.isValid():
            self.color = c
            self.update_style()

    def update_style(self):
        hex_c = self.color.name()
        # Calculate contrast text color
        text_c = "black" if self.color.lightness() > 128 else "white"
        self.setStyleSheet(f"background-color: {hex_c}; color: {text_c}; font-weight: bold; border: 1px solid #555;")
        self.setText(hex_c.upper())

    def get_hex(self):
        return self.color.name()

# --- UTILITY: Account Import Dialog ---
class AdvancedAccountDialog(QDialog):
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setWindowTitle("Import Accounts")
        self.resize(500, 400)
        layout = QVBoxLayout(self)
        layout.addWidget(QLabel("Paste Account Names (One per line):"))
        self.text_area = QTextEdit()
        layout.addWidget(self.text_area)
        btn_box = QHBoxLayout()
        imp_btn = QPushButton("Import Text File")
        imp_btn.clicked.connect(self.import_file)
        ok_btn = QPushButton("Use These Accounts")
        ok_btn.clicked.connect(self.accept)
        btn_box.addWidget(imp_btn)
        btn_box.addWidget(ok_btn)
        layout.addLayout(btn_box)

    def import_file(self):
        path, _ = QFileDialog.getOpenFileName(self, "Select File", "", "Text (*.txt *.csv)")
        if path:
            try:
                with open(path, 'r', encoding='utf-8') as f:
                    self.text_area.setText(f.read().replace(',', '\n'))
            except Exception as e:
                QMessageBox.warning(self, "Error", str(e))

    def get_accounts(self):
        raw = self.text_area.toPlainText()
        return [x.strip() for x in raw.split('\n') if x.strip()]

# --- WORKER THREAD ---
class ProcessingThread(QThread):
    progress = pyqtSignal(int)
    status = pyqtSignal(str)
    finished = pyqtSignal(str)
    error = pyqtSignal(str)

    def __init__(self, params):
        super().__init__()
        self.p = params

    def run(self):
        con = None
        try:
            p = self.p
            self.progress.emit(5)
            self.status.emit("Initializing DuckDB engine...")

            # 1. Connect and Load Data
            con = duckdb.connect(database=':memory:')
            
            # Helper to sanitize column names for SQL
            def safe_col(c): return f'"{c}"'

            if p['file_type'] == 'csv':
                con.execute(f"CREATE OR REPLACE VIEW raw_data AS SELECT * FROM read_csv_auto('{p['file_path']}', all_varchar=True);")
            else:
                # Load via Pandas to handle specific sheet
                df_source = pd.read_excel(p['file_path'], sheet_name=p['sheet_name'], dtype=str)
                con.register('df_source', df_source)
                con.execute("CREATE OR REPLACE VIEW raw_data AS SELECT * FROM df_source;")

            self.progress.emit(15)
            self.status.emit("Preparing data structure...")

            # 2. Prepare Calculation Logic
            doc_col = safe_col(p['doc_col'])
            acc_col = safe_col(p['acc_col'])
            parent_col_sql = safe_col(p['parent_col']) if p['use_parent'] else "NULL"
            
            if p['mode'] == 'debit_credit':
                net_calc = f'COALESCE(TRY_CAST("{p["debit_col"]}" AS DOUBLE), 0) - COALESCE(TRY_CAST("{p["credit_col"]}" AS DOUBLE), 0)'
                # For totals
                dr_calc = f'COALESCE(TRY_CAST("{p["debit_col"]}" AS DOUBLE), 0)'
                cr_calc = f'COALESCE(TRY_CAST("{p["credit_col"]}" AS DOUBLE), 0)'
            else:
                net_calc = f'COALESCE(TRY_CAST("{p["net_col"]}" AS DOUBLE), 0)'
                # Approximate Dr/Cr for summary if only Net is given
                dr_calc = f'CASE WHEN ({net_calc}) > 0 THEN ({net_calc}) ELSE 0 END'
                cr_calc = f'CASE WHEN ({net_calc}) < 0 THEN ABS({net_calc}) ELSE 0 END'

            # 3. Create Clean View
            con.execute(f"""
                CREATE OR REPLACE VIEW clean_view AS
                SELECT 
                    *,
                    {doc_col} AS _doc_id,
                    {acc_col} AS _acc_name,
                    {parent_col_sql} AS _parent_grp,
                    {net_calc} AS _net_val,
                    {dr_calc} AS _dr_val,
                    {cr_calc} AS _cr_val
                FROM raw_data
            """)

            # 4. Filter logic - Fix for quote escaping
            target_accounts = p['target_accounts']
            # Pythonic way to handle SQL string list without backslash issues
            escaped_accs = [x.replace("'", "''") for x in target_accounts]
            target_sql_list = ", ".join([f"'{x}'" for x in escaped_accs])
            
            self.status.emit("Identifying relevant documents...")
            self.progress.emit(25)
            
            # Find all Docs that contain at least one of the selected accounts
            relevant_docs_query = f"SELECT DISTINCT _doc_id FROM clean_view WHERE _acc_name IN ({target_sql_list})"

            # 5. Parent Grouping & Totals Map (Global for the selected accounts)
            parent_map = {}
            totals_map = {}
            
            # A. Get Parent Groupings for ALL accounts (so column headers work for contra-accounts too)
            # Use DISTINCT to be fast
            parent_sql = "SELECT DISTINCT _acc_name, _parent_grp FROM clean_view WHERE _parent_grp IS NOT NULL"
            all_parents = con.execute(parent_sql).fetchall()
            for r in all_parents:
                acc_name, grp_name = r
                parent_map[acc_name] = grp_name

            # B. Get Totals ONLY for Selected Target Accounts (For the Index Sheet)
            stats_sql = f"""
                SELECT _acc_name, SUM(_dr_val), SUM(_cr_val)
                FROM clean_view
                WHERE _acc_name IN ({target_sql_list})
                GROUP BY _acc_name
            """
            stats = con.execute(stats_sql).fetchall()
            for row in stats:
                acc, dr, cr = row
                totals_map[acc] = (dr or 0.0, cr or 0.0)

            # 6. PIVOT Execution
            self.status.emit("Pivoting data (this may take time)...")
            self.progress.emit(40)

            # Index columns (Context columns)
            index_cols = p['export_cols']
            index_cols_sql = ", ".join([safe_col(c) for c in index_cols])

            pivot_sql = f"""
                PIVOT (
                    SELECT {index_cols_sql}, _acc_name, _net_val
                    FROM clean_view
                    WHERE _doc_id IN ({relevant_docs_query})
                )
                ON _acc_name
                USING SUM(_net_val)
                GROUP BY {index_cols_sql}
            """
            
            df_pivot = con.execute(pivot_sql).fetchdf()
            
            if df_pivot.empty:
                raise Exception("No data found matching the criteria.")

            con.close() # Close DB to free memory

            # 7. Post-Processing & Export
            self.status.emit("Structuring Excel report...")
            self.progress.emit(60)

            # Columns identification
            all_cols = list(df_pivot.columns)
            pivot_acc_cols = [c for c in all_cols if c not in index_cols]
            
            # Fill NaN with 0 for calculation, but keep index cols as is
            # We only want to fill 0 in account columns to check relevance
            # But efficiently:
            df_pivot[pivot_acc_cols] = df_pivot[pivot_acc_cols].fillna(0)

            base_name, ext = os.path.splitext(p['output_path'])
            
            # Determine batches
            chunk_size = p['sheet_limit'] if p['split_files'] else len(target_accounts)
            # Filter selected accounts that actually exist in the pivot data
            valid_targets = [acc for acc in target_accounts if acc in pivot_acc_cols]
            
            if not valid_targets:
                 raise Exception("None of the selected accounts appeared in the generated pivot data.")

            batches = [valid_targets[i:i + chunk_size] for i in range(0, len(valid_targets), chunk_size)]
            
            total_batches = len(batches)
            
            for b_idx, batch_accounts in enumerate(batches):
                
                # Determine filename
                if total_batches > 1:
                    save_path = f"{base_name}_Part{b_idx+1}{ext}"
                else:
                    save_path = p['output_path']

                self.status.emit(f"Writing file {b_idx+1} of {total_batches}...")
                
                workbook = xlsxwriter.Workbook(save_path)
                
                # Formats
                fmt_header = workbook.add_format({
                    'bold': True, 'fg_color': p['color_header'], 'border': 1, 'text_wrap': True, 'valign': 'vcenter', 'align': 'center'
                })
                fmt_parent = workbook.add_format({
                    'bold': True, 'fg_color': '#E0E0E0', 'border': 1, 'align': 'center', 'valign': 'vcenter'
                })
                fmt_total = workbook.add_format({
                    'bold': True, 'bg_color': p['color_total'], 'border': 1, 'num_format': '#,##0.00'
                })
                fmt_num = workbook.add_format({'num_format': '#,##0.00'})
                fmt_link = workbook.add_format({'font_color': 'blue', 'underline': 1})
                
                # --- INDEX SHEET ---
                ws_idx = workbook.add_worksheet("Index")
                ws_idx.set_column(0, 0, 8)  # S.No
                ws_idx.set_column(1, 1, 40) # Account Name
                ws_idx.set_column(2, 4, 15) # Parent, Dr, Cr
                
                idx_headers = ["S.No", "Account Name", "Parent Grouping", "Total Debit", "Total Credit", "Link"]
                ws_idx.write_row(0, 0, idx_headers, fmt_header)
                
                s_no = 1
                
                for acc_name in batch_accounts:
                    # 1. Filter Data for this specific account
                    # Logic: Get rows where this account is non-zero
                    # Then, in that subset, identify relevant columns (non-zero columns)
                    
                    subset = df_pivot[df_pivot[acc_name] != 0].copy()
                    if subset.empty: continue
                    
                    # 2. Filter relevant columns: Index Cols + (Cols with sum != 0 in this subset)
                    # Exclude index cols from check
                    numeric_subset = subset[pivot_acc_cols]
                    # Identify columns that are not all zero
                    # (This is the feature: "Relevant columns only")
                    relevant_accs = numeric_subset.columns[(numeric_subset != 0).any()].tolist()
                    
                    # Ensure the main account is in relevant list (it should be, but safety first)
                    if acc_name not in relevant_accs: relevant_accs.append(acc_name)
                    
                    # Sort relevant accounts: Main account first, then others alphabetically
                    relevant_accs.sort()
                    if acc_name in relevant_accs:
                        relevant_accs.remove(acc_name)
                        relevant_accs.insert(0, acc_name)
                        
                    final_cols = index_cols + relevant_accs
                    final_df = subset[final_cols]
                    
                    # 3. Write Sheet
                    sheet_name = str(s_no) # Keep short
                    ws = workbook.add_worksheet(sheet_name)
                    
                    # Write Parent Grouping Header (Optional)
                    row_offset = 0
                    if p['show_parent_header']:
                        row_offset = 1
                        # Write merged parent headers
                        # Index cols get empty parent
                        for col_idx, col_name in enumerate(final_cols):
                            if col_name in index_cols:
                                ws.write(0, col_idx, "", fmt_parent)
                            else:
                                p_grp = parent_map.get(col_name, "")
                                ws.write(0, col_idx, p_grp, fmt_parent)

                    # Write Main Header
                    ws.write_row(row_offset, 0, final_cols, fmt_header)
                    
                    # Write Data (Using XlsxWriter loop is slow for big data, use direct lists)
                    # Convert to list of lists for speed
                    data_rows = final_df.values.tolist()
                    
                    start_data_row = row_offset + 1
                    for r_idx, row_data in enumerate(data_rows):
                        ws.write_row(start_data_row + r_idx, 0, row_data)
                    
                    # Formatting & Totals
                    last_row = start_data_row + len(data_rows)
                    ws.write(last_row, 0, "TOTAL", fmt_total)
                    
                    # Freeze Panes
                    ws.freeze_panes(start_data_row, len(index_cols))
                    
                    for c_idx, col_name in enumerate(final_cols):
                        # Width
                        ws.set_column(c_idx, c_idx, max(len(str(col_name)), 12))
                        
                        # Number Format & Total Formula (Only for account columns)
                        if col_name not in index_cols:
                            # Apply number format to data range
                            # Note: set_column applies to whole column, but we want headers distinct
                            # Xlsxwriter applies formatting efficiently via set_column
                            ws.set_column(c_idx, c_idx, 15, fmt_num)
                            
                            # Write Sum Formula
                            col_char = xlsxwriter.utility.xl_col_to_name(c_idx)
                            formula = f"=SUM({col_char}{start_data_row+1}:{col_char}{last_row})"
                            ws.write_formula(last_row, c_idx, formula, fmt_total)

                    # 4. Update Index Sheet
                    dr, cr = totals_map.get(acc_name, (0,0))
                    p_grp = parent_map.get(acc_name, "")
                    
                    ws_idx.write(s_no, 0, s_no)
                    ws_idx.write(s_no, 1, acc_name)
                    ws_idx.write(s_no, 2, p_grp)
                    ws_idx.write(s_no, 3, dr, fmt_num)
                    ws_idx.write(s_no, 4, cr, fmt_num)
                    # Hyperlink
                    ws_idx.write_url(s_no, 5, f"internal:'{sheet_name}'!A1", string=f"Go to Sheet {sheet_name}", cell_format=fmt_link)
                    
                    s_no += 1
                
                workbook.close()
                self.progress.emit(60 + int(30 * (b_idx+1)/total_batches))

            self.progress.emit(100)
            self.finished.emit(f"Successfully processed {total_batches} file(s).")

        except Exception as e:
            if con: con.close()
            import traceback
            traceback.print_exc()
            self.error.emit(str(e))

# --- MAIN UI ---
class JournalApp(QMainWindow):
    def __init__(self):
        super().__init__()
        self.setWindowTitle("JV Dump to Columnar Convert")
        self.resize(1100, 900)
        self.df_cols = []
        self.file_path = ""
        self.sheet_name = ""
        self.init_ui()

    def init_ui(self):
        main_widget = QWidget()
        self.setCentralWidget(main_widget)
        main_layout = QVBoxLayout(main_widget)
        
        # Scroll Area for settings
        scroll = QScrollArea()
        scroll.setWidgetResizable(True)
        content = QWidget()
        self.form_layout = QVBoxLayout(content)
        scroll.setWidget(content)
        main_layout.addWidget(scroll)

        # 1. File Selection
        self.add_section("1. Data Source", self.ui_file_selection)
        # 2. Mapping
        self.add_section("2. Column Mapping", self.ui_mapping)
        # 3. Exports
        self.add_section("3. Columns to Export (Context)", self.ui_export_cols)
        # 4. Accounts
        self.add_section("4. Target Accounts", self.ui_accounts)
        # 5. Settings
        self.add_section("5. Configuration & Styling", self.ui_settings)

        # Action Bar
        action_bar = QHBoxLayout()
        self.btn_run = QPushButton("Generate Reports")
        self.btn_run.setStyleSheet("background-color: #0078D7; color: white; font-size: 14pt; padding: 10px; font-weight: bold;")
        self.btn_run.clicked.connect(self.run_process)
        action_bar.addWidget(self.btn_run)
        
        main_layout.addLayout(action_bar)
        
        self.progress = QProgressBar()
        self.progress.setVisible(False)
        main_layout.addWidget(self.progress)
        
        self.status = QStatusBar()
        self.setStatusBar(self.status)

    def add_section(self, title, func):
        grp = QGroupBox(title)
        grp.setStyleSheet("QGroupBox { font-weight: bold; font-size: 11pt; margin-top: 10px; } QGroupBox::title { subcontrol-origin: margin; left: 10px; padding: 0 3px; }")
        lyt = QVBoxLayout()
        func(lyt)
        grp.setLayout(lyt)
        self.form_layout.addWidget(grp)

    # --- UI Builders ---
    def ui_file_selection(self, layout):
        h = QHBoxLayout()
        self.txt_file = QLineEdit("No file selected")
        self.txt_file.setReadOnly(True)
        btn = QPushButton("Browse")
        btn.clicked.connect(self.browse_file)
        h.addWidget(self.txt_file)
        h.addWidget(btn)
        layout.addLayout(h)

    def ui_mapping(self, layout):
        fl = QFormLayout()
        self.cb_doc = QComboBox()
        self.cb_acc = QComboBox()
        fl.addRow("Voucher/Doc Identifier:", self.cb_doc)
        fl.addRow("Account/Ledger Name:", self.cb_acc)
        
        # Value Mode
        self.tabs_val = QTabWidget()
        w_dc = QWidget()
        l_dc = QFormLayout(w_dc)
        self.cb_dr = QComboBox()
        self.cb_cr = QComboBox()
        l_dc.addRow("Debit Column:", self.cb_dr)
        l_dc.addRow("Credit Column:", self.cb_cr)
        
        w_net = QWidget()
        l_net = QFormLayout(w_net)
        self.cb_net = QComboBox()
        l_net.addRow("Net Value Column:", self.cb_net)
        
        self.tabs_val.addTab(w_dc, "Debit & Credit")
        self.tabs_val.addTab(w_net, "Single Net Column")
        fl.addRow(self.tabs_val)
        
        # Parent Group
        self.chk_parent = QCheckBox("Source has Parent Grouping Column")
        self.chk_parent.toggled.connect(lambda c: self.cb_parent.setEnabled(c))
        self.cb_parent = QComboBox()
        self.cb_parent.setEnabled(False)
        fl.addRow(self.chk_parent, self.cb_parent)
        
        layout.addLayout(fl)
        
        # Signals to refresh export exclusion
        for cb in [self.cb_doc, self.cb_acc, self.cb_dr, self.cb_cr, self.cb_net, self.cb_parent]:
            cb.currentIndexChanged.connect(self.refresh_export_cols)

    def ui_export_cols(self, layout):
        layout.addWidget(QLabel("Select columns to keep in the report (Date, Narration, etc.):"))
        self.lst_export = QListWidget()
        self.lst_export.setSelectionMode(QAbstractItemView.ExtendedSelection)
        self.lst_export.setFixedHeight(120)
        layout.addWidget(self.lst_export)

    def ui_accounts(self, layout):
        h = QHBoxLayout()
        self.txt_search = QLineEdit()
        self.txt_search.setPlaceholderText("Search Accounts...")
        self.txt_search.textChanged.connect(self.filter_accounts)
        btn_adv = QPushButton("Import List / Advanced")
        btn_adv.clicked.connect(self.open_import_dialog)
        h.addWidget(self.txt_search)
        h.addWidget(btn_adv)
        layout.addLayout(h)
        
        self.lst_accs = QListWidget()
        self.lst_accs.setSelectionMode(QAbstractItemView.ExtendedSelection)
        self.lst_accs.setFixedHeight(200)
        layout.addWidget(self.lst_accs)
        
        self.lbl_sel_count = QLabel("0 Selected")
        self.lst_accs.itemSelectionChanged.connect(lambda: self.lbl_sel_count.setText(f"{len(self.lst_accs.selectedItems())} Selected"))
        layout.addWidget(self.lbl_sel_count)

    def ui_settings(self, layout):
        # styling
        gb_style = QGroupBox("Report Styling")
        ls = QHBoxLayout()
        self.col_head = ColorButton(QColor("#D7E4BC"))
        self.col_total = ColorButton(QColor("#FFFF00"))
        ls.addWidget(QLabel("Header Color:"))
        ls.addWidget(self.col_head)
        ls.addWidget(QLabel("Total Row Color:"))
        ls.addWidget(self.col_total)
        gb_style.setLayout(ls)
        layout.addWidget(gb_style)
        
        # Limits and Features
        gb_opt = QGroupBox("Export Options")
        lo = QFormLayout()
        
        self.chk_parent_header = QCheckBox("Show Parent Grouping as Top Header Row")
        lo.addRow(self.chk_parent_header)
        
        self.chk_split = QCheckBox("Split into multiple Excel files (Workbook Limit)")
        self.chk_split.toggled.connect(lambda c: self.spin_limit.setEnabled(c))
        self.spin_limit = QSpinBox()
        self.spin_limit.setRange(1, 500)
        self.spin_limit.setValue(50)
        self.spin_limit.setEnabled(False)
        lo.addRow(self.chk_split)
        lo.addRow("Sheets per Workbook:", self.spin_limit)
        
        gb_opt.setLayout(lo)
        layout.addWidget(gb_opt)

    # --- Logic ---
    def browse_file(self):
        path, _ = QFileDialog.getOpenFileName(self, "Select File", "", "Excel/CSV (*.xlsx *.xls *.csv)")
        if not path: return
        self.file_path = path
        self.txt_file.setText(path)
        
        if path.endswith(('.xlsx', '.xls')):
            try:
                xl = pd.ExcelFile(path)
                sheets = xl.sheet_names
                if len(sheets) > 1:
                    from PyQt5.QtWidgets import QInputDialog
                    item, ok = QInputDialog.getItem(self, "Select Sheet", "Sheet:", sheets, 0, False)
                    if ok: self.sheet_name = item
                    else: return
                else:
                    self.sheet_name = sheets[0]
                
                # Peek for columns
                df = pd.read_excel(path, sheet_name=self.sheet_name, nrows=1)
                self.df_cols = list(df.columns)
            except Exception as e:
                QMessageBox.critical(self, "Error", f"Read Error: {e}")
                return
        else:
            # CSV peek via DuckDB
            try:
                con = duckdb.connect()
                con.execute(f"CREATE VIEW t AS SELECT * FROM read_csv_auto('{path}', all_varchar=True) LIMIT 1")
                self.df_cols = [x[0] for x in con.execute("DESCRIBE t").fetchall()]
                con.close()
            except Exception as e:
                QMessageBox.critical(self, "Error", f"CSV Error: {e}")
                return

        self.populate_ui_data()

    def populate_ui_data(self):
        # Update Combos
        combos = [self.cb_doc, self.cb_acc, self.cb_dr, self.cb_cr, self.cb_net, self.cb_parent]
        for c in combos:
            c.clear()
            c.addItems(self.df_cols)
        
        # Auto-Guess
        for i, col in enumerate(self.df_cols):
            cl = col.lower()
            if 'voucher' in cl or 'guid' in cl: self.cb_doc.setCurrentIndex(i)
            if 'ledger' in cl or 'acc' in cl and 'name' in cl: self.cb_acc.setCurrentIndex(i)
            if 'debit' in cl: self.cb_dr.setCurrentIndex(i)
            if 'credit' in cl: self.cb_cr.setCurrentIndex(i)
            if 'net' in cl: self.cb_net.setCurrentIndex(i)
            if 'group' in cl or 'parent' in cl: 
                self.cb_parent.setCurrentIndex(i)
                self.chk_parent.setChecked(True)

        self.refresh_export_cols()
        self.load_accounts()

    def load_accounts(self):
        self.status.showMessage("Loading unique accounts...")
        QApplication.processEvents()
        
        acc_col = self.cb_acc.currentText()
        if not acc_col: return
        
        try:
            con = duckdb.connect()
            # Handle quotes in col name
            col_sql = f'"{acc_col}"'
            
            if self.file_path.endswith('.csv'):
                q = f"SELECT DISTINCT {col_sql} FROM read_csv_auto('{self.file_path}', all_varchar=True) ORDER BY 1"
                res = con.execute(q).fetchall()
                accs = [r[0] for r in res if r[0]]
            else:
                # Excel: Use pandas for distinct to be safe with types
                df = pd.read_excel(self.file_path, sheet_name=self.sheet_name, usecols=[acc_col], dtype=str)
                accs = sorted(df[acc_col].dropna().unique().tolist())
            
            self.lst_accs.clear()
            self.lst_accs.addItems(accs)
            con.close()
            self.status.showMessage("Ready")
        except Exception as e:
            QMessageBox.warning(self, "Error", str(e))

    def filter_accounts(self, txt):
        for i in range(self.lst_accs.count()):
            it = self.lst_accs.item(i)
            it.setHidden(txt.lower() not in it.text().lower())

    def refresh_export_cols(self):
        # Determine excluded columns based on mapping
        excluded = {self.cb_doc.currentText(), self.cb_acc.currentText()}
        if self.tabs_val.currentIndex() == 0: # Dr/Cr
            excluded.update([self.cb_dr.currentText(), self.cb_cr.currentText()])
        else:
            excluded.add(self.cb_net.currentText())
        
        if self.chk_parent.isChecked():
            excluded.add(self.cb_parent.currentText())
            
        self.lst_export.clear()
        for c in self.df_cols:
            if c not in excluded:
                it = self.lst_export.addItem(c)
                # Auto select non-system columns
                if not c.startswith('$'): 
                    self.lst_export.item(self.lst_export.count()-1).setSelected(True)

    def open_import_dialog(self):
        dlg = AdvancedAccountDialog(self)
        if dlg.exec_():
            imported = dlg.get_accounts()
            # Select matches
            found = 0
            for i in range(self.lst_accs.count()):
                it = self.lst_accs.item(i)
                if it.text() in imported:
                    it.setSelected(True)
                    found += 1
            QMessageBox.information(self, "Import", f"Selected {found} accounts found in the list.")

    def run_process(self):
        # Validate
        if not self.file_path: return
        targets = [x.text() for x in self.lst_accs.selectedItems()]
        if not targets:
            QMessageBox.warning(self, "Error", "Select at least one account.")
            return
        
        export_cols = [x.text() for x in self.lst_export.selectedItems()]
        
        out_path, _ = QFileDialog.getSaveFileName(self, "Save Report", "Columnar_Report.xlsx", "Excel Files (*.xlsx)")
        if not out_path: return

        # Params
        params = {
            'file_path': self.file_path,
            'file_type': 'csv' if self.file_path.lower().endswith('.csv') else 'excel',
            'sheet_name': self.sheet_name,
            'output_path': out_path,
            'doc_col': self.cb_doc.currentText(),
            'acc_col': self.cb_acc.currentText(),
            'mode': 'debit_credit' if self.tabs_val.currentIndex() == 0 else 'net',
            'debit_col': self.cb_dr.currentText(),
            'credit_col': self.cb_cr.currentText(),
            'net_col': self.cb_net.currentText(),
            'use_parent': self.chk_parent.isChecked(),
            'parent_col': self.cb_parent.currentText(),
            'target_accounts': targets,
            'export_cols': export_cols,
            # Styling
            'color_header': self.col_head.get_hex(),
            'color_total': self.col_total.get_hex(),
            # Features
            'show_parent_header': self.chk_parent_header.isChecked(),
            'split_files': self.chk_split.isChecked(),
            'sheet_limit': self.spin_limit.value()
        }
        
        self.progress.setVisible(True)
        self.progress.setValue(0)
        self.btn_run.setEnabled(False)
        
        self.worker = ProcessingThread(params)
        self.worker.progress.connect(self.progress.setValue)
        self.worker.status.connect(self.status.showMessage)
        self.worker.finished.connect(self.on_success)
        self.worker.error.connect(self.on_error)
        self.worker.start()

    def on_success(self, msg):
        self.progress.setVisible(False)
        self.btn_run.setEnabled(True)
        self.status.showMessage("Done.")
        QMessageBox.information(self, "Success", msg)

    def on_error(self, err):
        self.progress.setVisible(False)
        self.btn_run.setEnabled(True)
        QMessageBox.critical(self, "Error", err)

if __name__ == '__main__':
    app = QApplication(sys.argv)
    app.setStyle("Fusion")
    
    # Optional: Tweak Palette for dark/light mode integration
    palette = QPalette()
    palette.setColor(QPalette.ButtonText, Qt.black)
    app.setPalette(palette)
    
    win = JournalApp()
    win.show()
    sys.exit(app.exec_())
